<?php
/**
 * @author      Buro RaDer (i.o.v. SIEL - Acumulus) https://burorader.com/
 * @copyright   SIEL BV https://www.siel.nl/acumulus/
 * @license     GPLv3
 */

declare(strict_types=1);

use Joomla\CMS\Factory;
use Joomla\CMS\Plugin\CMSPlugin;
use Siel\Acumulus\Api;
use Siel\Acumulus\Collectors\PropertySources;
use Siel\Acumulus\Data\Invoice;
use Siel\Acumulus\Data\Line;
use Siel\Acumulus\Data\LineType;
use Siel\Acumulus\Fld;
use Siel\Acumulus\Helpers\Container;
use Siel\Acumulus\Helpers\Message;
use Siel\Acumulus\Helpers\Severity;
use Siel\Acumulus\Invoice\InvoiceAddResult;
use Siel\Acumulus\Invoice\Source;
use Siel\Acumulus\Meta;
use Siel\Joomla\Component\Acumulus\Administrator\Extension\AcumulusComponent;

/**
 * The AcumulusCustomiseInvoice plugin class contains plumbing and example code
 * to react to events triggered by the Acumulus component. These events allow you
 * to:
 * - Prevent sending an invoice to Acumulus.
 * - Customise the invoice before, during, or after it is being created.
 * - Process the results of sending the invoice to Acumulus.
 * - Inject custom actions at any of the event moments.
 *
 * Usage of this plugin:
 * You can use and modify this example plugin as you like:
 * - only implement the events you are going to use.
 * - add your own event handling in those handler methods.
 *
 * Documentation for the events:
 *
 * The Acumulus module defines the following hooks:
 * 1) {@see \Siel\Acumulus\Joomla\Helpers\Event::triggerInvoiceCreateBefore() onAcumulusInvoiceCreateBefore}
 * 2) {@see \Siel\Acumulus\Joomla\Helpers\Event::triggerLineCollectBefore() onAcumulusLineCollectBefore}
 * 3) {@see \Siel\Acumulus\Joomla\Helpers\Event::triggerLineCollectAfter() onAcumulusLineCollectAfter}
 * 4) {@see \Siel\Acumulus\Joomla\Helpers\Event::triggerInvoiceCollectAfter() onAcumulusInvoiceCollectAfter}
 * 5) {@see \Siel\Acumulus\Joomla\Helpers\Event::triggerInvoiceCreateAfter() onAcumulusInvoiceCreateAfter}
 * 6) {@see \Siel\Acumulus\Joomla\Helpers\Event::triggerInvoiceSendBefore() onAcumulusInvoiceSendBefore}
 * 7) {@see \Siel\Acumulus\Joomla\Helpers\Event::triggerInvoiceSendAfter() onAcumulusInvoiceSendAfter}
 *
 * The hooks are documented in the {@see \Siel\Acumulus\PrestaShop\Helpers\Event Event interface}.
 * The links in the list above point to that documentation, but you probably need
 * generated PHPDoc documentation or a good IDE to follow those links. If not navigable,
 * go to the file {webroot}/modules/acumulus/vendor/siel/acumulus/src/Helpers/Event.php.
 *
 * External Resources:
 *
 * - https://apidoc.sielsystems.nl/content/invoice-add.
 * - https://apidoc.sielsystems.nl/content/warning-error-and-status-response-section-most-api-calls
 *
 * @noinspection PhpUnused
 */
class PlgAcumulusAcumulusCustomiseInvoice extends CMSPlugin
{
    /**
     * Do not access directly, use getAcumulusContainer().
     */
    private Container $acumulusContainer;

    private function getAcumulusComponent(): AcumulusComponent
    {
        /** @noinspection PhpUnhandledExceptionInspection */
        /** @noinspection PhpIncompatibleReturnTypeInspection */
        return Factory::getApplication()->bootComponent('acumulus');
    }

    /**
     * Do not call directly, use getAcumulusContainer().
     */
    private function init(): void
    {
        if (!isset($this->acumulusContainer)) {
            $this->acumulusContainer = $this->getAcumulusComponent()->getAcumulusModel()->getAcumulusContainer();
        }
    }

    /**
     * Returns the {@see Container Acumulus container}, so this custom plugin has access
     * to the Acumulus classes, configuration, translations, and constants.
     */
    private function getAcumulusContainer(): Container
    {
        $this->init();
        return $this->acumulusContainer;
    }

    /**
     * Helper method to translate strings (from the Acumulus plugin).
     *
     * @param string $key
     *  The key to get a translation for.
     *
     * @return string
     *   The translation for the given key or the key itself if no translation could be
     *   found.
     */
    private function t(string $key): string
    {
        return $this->getAcumulusContainer()->getTranslator()->get($key);
    }

    /**
     * Event observer to react just before an invoice will be created.
     *
     * This event allows you to:
     * - Prevent the invoice from being created and sent at all. To do so,
     *   change the send-status using {@see InvoiceAddResult::setSendStatus()}
     *   on the $localResult parameter.
     * - Inject custom behaviour before the invoice is created (collected and
     *   completed) and sent.
     *
     * @param Source $invoiceSource
     *   The source object (order, credit note) for which the invoice was created.
     * @param InvoiceAddResult $localResult
     *   Contains any earlier generated messages and the initial send-status.
     *   You can add your own messages and/or change the send-status.
     */
    public function onAcumulusInvoiceCreateBefore(Source $invoiceSource, InvoiceAddResult $localResult): void
    {
        $this->getAcumulusContainer()->getLog()->debug(__METHOD__ . '(order_id=%d)', $invoiceSource->getId());
        try {
            // Do something.
        } catch (Throwable $e) {
            // Prevent creating and sending the invoice:
            $localResult->setSendStatus(InvoiceAddResult::NotSent_LocalErrors);
            $localResult->addMessage(Message::createFromException($e));
            // or just your own error message:
            $localResult->addMessage(Message::create('My message', Severity::Error));
        }
    }

    /**
     * Event observer to react just before a main line and its possible children will be
     * collected.
     *
     * This event allows you to:
     * - Add property sources to the $collectorManager.
     * - Directly set some values or metadata on the line.
     * - Inject custom behaviour before a line is collected.
     * - Prevent a line being collected and added to the invoice by setting the metadata
     *   value {@see \Siel\Acumulus\Meta::DoNotAdd} to true:
     *   <code>$line->metadataSet(Meta::DoNotAdd, true);</code>
     *
     * @param \Siel\Acumulus\Data\Line $line
     *   The line to store the collected values in.
     * @param \Siel\Acumulus\Collectors\PropertySources $propertySources
     *   The set of "objects" that can be used to collect data from. Available will be:
     *   - 'invoice': The {@see \Siel\Acumulus\Data\Invoice} being collected. The invoice,
     *     customer, both addresses, and emailAsPdf parts will already have been
     *     collected.
     *   - 'source': The {@see \Siel\Acumulus\Invoice\Source} for which the invoice is
     *     being collected.
     *   - '{lineType}LineInfo': the main "object" that results in 1 line. E.g. a product
     *     order item record. Each record should result in 1 line, whereas the above
     *     "objects" (invoice and source) are the same for every line being collected.
     *     {lineType} is the {@see \lcfirst()} value of the
     *     {@see \Siel\Acumulus\Data\LineType} constant name (not its value).
     *   - 'key': The key with which the above "lineInfo" object was passed to the
     *     collector. Normally, this is not needed, but there are cases where it contains
     *     valuable information (OC: shipping tax lines).
     */
    public function onAcumulusLineCollectBefore(Line $line, PropertySources $propertySources): void
    {
        /** @var \Siel\Acumulus\Invoice\Source $source */
        $source = $propertySources->get('source');
        /** @var \Siel\Acumulus\Data\Invoice $invoice */
        $invoice = $propertySources->get('invoice');

        // Here you can already add values to the line to be collected or add or change
        // some propertySources.
        $this->getAcumulusContainer()->getLog()->debug(__METHOD__ . '(type=%s, order-id=%d)', $line->getType(), $source->getId());
        $someValue = random_int(1, 100);
        $propertySources->add('myObject', $someValue);
        $someCondition = $someValue === 0;
        if ($someCondition) {
            // Do not add this line.
            $line->metadataSet(Meta::DoNotAdd, true);
        }
    }

    /**
     * Event observer to react after a main line and its possible children have been
     * "collected".
     *
     * This event allows you to:
     * - Remove property sources added during the {@see triggerLineCollectBefore}
     *   event.
     * - Prevent a line, and its children, from being added to the invoice by setting the
     *   metadata value {@see \Siel\Acumulus\Meta::DoNotAdd} to true:
     *   <code>$line->metadataSet(Meta::DoNotAdd, true);</code>
     * - Inject custom behaviour after a line, and its children, have been collected.
     *
     * @param \Siel\Acumulus\Data\Line $line
     *   The collected line.
     * @param \Siel\Acumulus\Collectors\PropertySources $propertySources
     *   The set of "objects" that can be used to collect data from.
     */
    public function onAcumulusLineCollectAfter(Line $line, PropertySources $propertySources): void
    {
        /** @var \Siel\Acumulus\Invoice\Source $source */
        $source = $propertySources->get('source');
        /** @var \Siel\Acumulus\Data\Invoice $invoice */
        $invoice = $propertySources->get('invoice');

        // Here you can make changes to the collected line based on your situation,
        // as well as remove added propertySources or prevent adding this line after all.
        $this->getAcumulusContainer()->getLog()->debug(__METHOD__ . '(type=%s, order-id=%d)', $line->getType(), $source->getId());
        $propertySources->remove('myObject');
    }

    /**
     * Event observer to react after an invoice for Acumulus has been "collected".
     *
     * This event allows you to:
     * - Change the invoice by changing the collected data. This is the place to do so if
     *   you need access to the data from the shop environment this library is running in.
     * - Prevent the invoice from being completed and sent. To do so, change the
     *   send-status using {@see InvoiceAddResult::setSendStatus()} on the
     *   $localResult parameter.
     * - Inject custom behaviour after the invoice has been created (collected),
     *   but before it is completed and sent.
     *
     * @param \Siel\Acumulus\Data\Invoice $invoice
     *   The invoice that has been collected.
     * @param Source $invoiceSource
     *   The source object (order, credit note) for which the invoice is created.
     * @param \Siel\Acumulus\Invoice\InvoiceAddResult $localResult
     *   Contains any earlier generated messages and the initial send-status.
     *   You can add your own messages and/or change the send-status.
     */
    public function onAcumulusInvoiceCollectAfter(Invoice $invoice, Source $invoiceSource, InvoiceAddResult $localResult): void
    {
        // Here you can make changes to the raw invoice based on your situation.
        $this->getAcumulusContainer()->getLog()->debug(__METHOD__ . '(order_id=%d)', $invoiceSource->getId());
    }

    /**
     * Event observer to react after an invoice for Acumulus has been created, that is
     * collected and completed.
     *
     * This event allows you to:
     * - Change the invoice by changing the collected and completed data. This is the
     *   place to do so if you need access to the data from the shop environment this
     *   library is running in.
     * - Prevent the invoice from being sent. To do so, change the send-status using
     *   {@see InvoiceAddResult::setSendStatus()} on the $localResult parameter.
     * - Inject custom behaviour after the invoice has been created, but before it is
     *   sent.
     *
     * @param \Siel\Acumulus\Data\Invoice $invoice
     *   The invoice that has been created.
     * @param Source $invoiceSource
     *   The source object (order, credit note) for which the invoice is created.
     * @param \Siel\Acumulus\Invoice\InvoiceAddResult $localResult
     *   Contains any earlier generated messages and the initial send-status.
     *   You can add your own messages and/or change the send-status.
     */
    public function onAcumulusInvoiceCreateAfter(Invoice $invoice, Source $invoiceSource, InvoiceAddResult $localResult): void
    {
        // Here you can make changes to the invoice based on your situation.
        $this->getAcumulusContainer()->getLog()->debug(__METHOD__ . '(order_id=%d)', $invoiceSource->getId());
    }

    /**
     * Event observer to react just before an invoice for Acumulus will be sent.
     *
     * This event allows you to:
     * - Change the invoice by adding or changing the collected data. This is
     *   the place to do so if you need access to the complete invoice itself
     *   just before sending. Note that no Shop order or credit note objects
     *   are passed to this event.
     * - Prevent the invoice from being sent. To do so, change the send-status
     *   using {@see InvoiceAddResult::setSendStatus()} on the $result
     *   parameter.
     * - Inject custom behaviour just before sending.
     *
     * @param \Siel\Acumulus\Data\Invoice $invoice
     *   The invoice that has been created.
     * @param \Siel\Acumulus\Invoice\InvoiceAddResult $localResult
     *   Contains any earlier generated messages and the initial send-status.
     *   You can add your own messages and/or change the send-status.
     */
    public function onAcumulusInvoiceSendBefore(Invoice $invoice, InvoiceAddResult $localResult): void
    {
        // Here you can make changes to the completed invoice based on your situation,
        $this->getAcumulusContainer()->getLog()->debug(__METHOD__ . '(order_id=%d)', $invoice->metadataGet(Meta::SourceId));
    }

    /**
     * Event observer to react to an invoice for Acumulus being sent.
     *
     * This event will also be triggered when sending the invoice resulted in an error,
     * but not when sending was prevented locally due to e.g. no need to send, an earlier
     * event that prevented sending, or the dry-run modus.
     *
     * This event allows you to:
     * - Inject custom behavior to react to the result of sending the invoice.
     *
     * @param \Siel\Acumulus\Data\Invoice $invoice
     *   The invoice that has been sent.
     * @param Source $invoiceSource
     *   The source object (order, credit note) for which the invoice was sent.
     * @param \Siel\Acumulus\Invoice\InvoiceAddResult $result
     *   The result, response, status, and any messages, as sent back by
     *   Acumulus (or set earlier locally).
     */
    public function onAcumulusInvoiceSendAfter(Invoice $invoice, Source $invoiceSource, InvoiceAddResult $result): void
    {
        // Here you can react to the result of sending the invoice to Acumulus
        $logMessage = sprintf(__METHOD__ . '(order_id=%d): ', $invoiceSource->getId());
        if ($result->getMessages(Severity::Exception)) {
            $this->getAcumulusContainer()->getLog()->debug($logMessage . $result->getMessages(Severity::Exception)[0]->getText());
        } elseif ($result->hasError()) {
            // Invoice was sent to Acumulus but not created due to (an) error(s)
            // in the invoice.
            $this->getAcumulusContainer()->getLog()->debug($logMessage . $result->getMessages(Severity::Error)[0]->getText());
        } else {
            // Sent successfully, invoice has been created in Acumulus:
            if ($result->getMessages(Severity::Warning)) {
                // With warnings.
                $this->getAcumulusContainer()->getLog()->debug($logMessage . $result->getMessages(Severity::Warning)[0]->getText());
            } else {
                // Without warnings.
                $this->getAcumulusContainer()->getLog()->debug($logMessage . 'success');
            }

            // Check if an entry id was created.
            $acumulusInvoice = $result->getMainApiResponse();
            if (!empty($acumulusInvoice['entryid'])) {
                // The invoice has been added.
                $this->getAcumulusContainer()->getLog()->debug($logMessage . 'invoice added');
                $invoiceNumber = $acumulusInvoice['invoicenumber'];
                $entryId = $acumulusInvoice['entryid'];
                $token = $acumulusInvoice['token'];
            } elseif (!empty($acumulusInvoice['conceptid'])) {
                // The invoice has been added as a concept.
                $this->getAcumulusContainer()->getLog()->debug($logMessage . 'concept added');
                $conceptId = $acumulusInvoice['conceptid'];
            } else {
                // The invoice was sent in test modus: no invoice nor a concept has been added.
                $this->getAcumulusContainer()->getLog()->debug($logMessage . 'sent in test modus');
            }
        }
    }
}
