<?php
/**
 * @noinspection AutoloadingIssuesInspection
 * @noinspection PhpIllegalPsrClassPathInspection
 */

declare(strict_types=1);

namespace Opencart\Catalog\Controller\Extension\AcumulusCustomiseInvoice\Module;

use Opencart\Admin\Controller\Extension\Acumulus\Module\Acumulus;
use Opencart\System\Engine\Controller;
use Siel\Acumulus\ApiClient\Acumulus as AcumulusApiClient;
use Siel\Acumulus\Collectors\PropertySources;
use Siel\Acumulus\Data\Invoice;
use Siel\Acumulus\Data\Line;
use Siel\Acumulus\Data\LineType;
use Siel\Acumulus\Fld;
use Siel\Acumulus\Helpers\Container;
use Siel\Acumulus\Helpers\Message;
use Siel\Acumulus\Helpers\Severity;
use Siel\Acumulus\Invoice\InvoiceAddResult;
use Siel\Acumulus\Invoice\Source;
use Siel\Acumulus\Meta;
use Siel\Acumulus\OpenCart\Helpers\Registry;
use Throwable;

use function sprintf;

/**
 * This 'catalog' controller is extended by the 'admin' controller from this extension!
 *
 * This 'catalog' controller contains:
 * - The properties used in 'catalog', and, optionally, in 'admin'.
 * - A constructor with initialising code.
 * - The event handling methods. The events can be triggered on both the 'admin' and
 *   'catalog' side. Therefore, they are defined in the 'catalog' controller and the
 *   'admin' controller inherits them from the 'catalog' controller.
 *
 * Additionally, the 'admin' controller contains:
 * - Installation code (registering the event handlers).
 * - Uninstall code (removing the registered event handlers).
 *
 * Usage of this extension:
 * - In this part, the 'catalog' controller, you should change the event handlers to let
 *   them do what you want to achieve and remove the event handlers you don't need.
 * - In the other part, the 'admin' controller, you should only register the events you
 *   are going to use and thus remove or comment out the events you are not going to use.
 *
 * Documentation for the events:
 *
 * The Acumulus module defines the following events:
 * 1) {@see \Siel\Acumulus\Helpers\Event::triggerInvoiceCreateBefore() invoiceCreateBefore}
 * 2) {@see \Siel\Acumulus\Helpers\Event::triggerLineCollectBefore() lineCollectBefore}
 * 3) {@see \Siel\Acumulus\Helpers\Event::triggerLineCollectAfter() lineCollectAfter}
 * 4) {@see \Siel\Acumulus\Helpers\Event::triggerInvoiceCollectAfter() invoiceCollectAfter}
 * 5) {@see \Siel\Acumulus\Helpers\Event::triggerInvoiceCreateAfter() invoiceCreateAfter}
 * 6) {@see \Siel\Acumulus\Helpers\Event::triggerInvoiceSendBefore() invoiceSendBefore}
 * 7) {@see \Siel\Acumulus\Helpers\Event::triggerInvoiceSendAfter() invoiceSendAfter}
 *
 * These events are documented in the {@see \Siel\Acumulus\Helpers\Event Event interface}.
 * The links in the list above point to that documentation, but you probably need
 * generated PHPDoc documentation or a good IDE to see them as links and be able to follow
 * those links. If not navigable, go to the file
 * {webroot}/modules/acumulus/vendor/siel/acumulus/src/Helpers/Event.php.
 *
 * External Resources:
 *
 * - https://apidoc.sielsystems.nl/content/invoice-add.
 * - https://apidoc.sielsystems.nl/content/warning-error-and-status-response-section-most-api-calls
 *
 * Examples:
 *
 * The documentation mentioned above and methods (event handlers) below contain some
 * example code to give yu an idea of what you can do, what information is available, etc.
 *
 * OpenCart 4 specific:
 *
 * Available properties (probably not exhaustive):
 *
 * @property \OpenCart\System\Library\Document $document
 * @property \OpenCart\System\Engine\Loader $load
 * @property \OpenCart\System\Library\Language $language
 * @property \OpenCart\System\Library\Session $session
 * @property \OpenCart\System\Library\Url $url
 * @property \OpenCart\System\Library\Response $response
 * @property \Opencart\Catalog\Model\Setting\Event|\Opencart\Admin\Model\Setting\Event $model_setting_event
 */
class AcumulusCustomiseInvoice extends Controller
{
    public function __construct(\Opencart\System\Engine\Registry $registry)
    {
        parent::__construct($registry);
        // Load Acumulus controller to ensure the Acumulus autoloader is loaded and that
        // a container is created.
        if (class_exists(Acumulus::class)) {
            new Acumulus($registry);
        }
    }

    /**
     * Returns the {@see Container Acumulus container}, so this custom plugin has access
     * to the Acumulus classes, configuration, translations, and constants.
     */
    protected function getAcumulusContainer(): Container
    {
        return Acumulus::getAcumulusContainer();
    }

    /**
     * Returns the {@see Registry Acumulus registry}, which is a wrapper around the
     * {@see \Opencart\System\Engine\Registry OpenCart registry} but with some additional
     * convenience methods.
     */
    protected function getAcumulusOcRegistry(): Registry
    {
        $this->getAcumulusContainer();
        return Registry::getInstance();
    }

    /**
     * Processes the event triggered before the start of the create process.
     */
    public function invoiceCreateBefore(Source $invoiceSource, InvoiceAddResult $localResult): void
    {
        $this->getAcumulusContainer()->getLog()->debug(__METHOD__ . '(order_id=%d)', $invoiceSource->getId());
        try {
            // Do something.
        } catch (Throwable $e) {
            // Prevent creating and sending the invoice:
            $localResult->setSendStatus(InvoiceAddResult::NotSent_LocalErrors);
            $localResult->addMessage(Message::createFromException($e));
            // or just your own error message:
            $localResult->addMessage(Message::create('My message', Severity::Error));
        }
    }

    /**
     * Processes the event triggered before a line gets collected.
     *
     * This event will be called once for each line regardless its line type.
     * Line type being one of the {@see \Siel\Acumulus\Data\LineType} constants.
     */
    public function lineCollectBefore(Line $line, PropertySources $propertySources): void
    {
        /** @var \Siel\Acumulus\Invoice\Source $source */
        $source = $propertySources->get('source');
        /** @var \Siel\Acumulus\Data\Invoice $invoice */
        $invoice = $propertySources->get('invoice');

        // Here you can already add values to the line to be collected or add or change
        // some propertySources.
        $this->getAcumulusContainer()->getLog()->debug(__METHOD__ . '(type=%s, order-id=%d)', $line->getType(), $source->getId());
        $someValue = random_int(1, 100);
        $propertySources->add('myObject', $someValue);
        $someCondition = $someValue === 0;
        if ($someCondition) {
            // Do not add this line.
            $line->metadataSet(Meta::DoNotAdd, true);
        }
    }

    /**
     * Processes the event triggered after a line got collected.
     *
     * This event will be called once for each line regardless its line type.
     * Line type being one of the {@see \Siel\Acumulus\Data\LineType} constants.
     */
    public function lineCollectAfter(Line $line, PropertySources $propertySources): void
    {
        /** @var \Siel\Acumulus\Invoice\Source $source */
        $source = $propertySources->get('source');
        /** @var \Siel\Acumulus\Data\Invoice $invoice */
        $invoice = $propertySources->get('invoice');

        // Here you can make changes to the collected line based on your situation,
        // as well as remove added propertySources or prevent adding this line after all.
        $this->getAcumulusContainer()->getLog()->debug(__METHOD__ . '(type=%s, order-id=%d)', $line->getType(), $source->getId());
        $propertySources->remove('myObject');
    }

    /**
     * Processes the event triggered after the raw invoice has been collected, but before
     * it will be completed.
     *
     * Note that the invoice needs yet to be completed:
     * - Fields that depend on configuration are yet to be filled in.
     * - Child lines are still hierarchical.
     * - Foreign amounts are not yet converted to euros.
     * - Amounts may be missing, e.g. only unitpriceinc' and 'vatamount' are filled in,
     *   while unitprice and vatrate are not yet derived.
     */
    public function invoiceCollectAfter(Invoice $invoice, Source $invoiceSource, InvoiceAddResult $localResult): void
    {
        // Here you can make changes to the raw invoice based on your situation.
        $this->getAcumulusContainer()->getLog()->debug(__METHOD__ . '(order_id=%d)', $invoiceSource->getId());
    }

    /**
     * Processes the event triggered after the invoice has been created,
     * that is: collected and completed.
     */
    public function invoiceCreateAfter(Invoice $invoice, Source $invoiceSource, InvoiceAddResult $localResult): void
    {
        // Here you can make changes to the invoice based on your situation.
        $this->getAcumulusContainer()->getLog()->debug(__METHOD__ . '(order_id=%d)', $invoiceSource->getId());
    }

    /**
     * Processes the event triggered before the invoice gets sent.
     *
     * Here you can:
     * - Make final changes to the invoice.
     * - Prevent sending the invoice after all.
     * - Inject custom behaviour (without changing the invoice or localResult) just before
     *   sending.
     */
    public function invoiceSendBefore(Invoice $invoice, InvoiceAddResult $localResult): void
    {
        // Here you can make changes to the completed invoice based on your situation,
        $this->getAcumulusContainer()->getLog()->debug(__METHOD__ . '(order_id=%d)', $invoice->metadataGet(Meta::SourceId));
    }

    /**
     * Processes the event triggered after an invoice has been sent to Acumulus.
     *
     *  Here you can:
     *  - React to the result of sending the invoice, e.g:
     *    - The token gives you access to the Acumulus invoice or packing slip PDF.
     *    - The entry-id can be used to correlate a payment to the invoice.
     *  - Inject custom behaviour (without changing the invoice or localResult) based on
     *    the result of the sending.
     */
    public function invoiceSendAfter(Invoice $invoice, Source $invoiceSource, InvoiceAddResult $result): void
    {
        // Here you can react to the result of sending the invoice to Acumulus
        $logMessage = sprintf(__METHOD__ . '(order_id=%d): ', $invoiceSource->getId());
        if ($result->getMessages(Severity::Exception)) {
            $this->getAcumulusContainer()->getLog()->debug($logMessage . $result->getMessages(Severity::Exception)[0]->getText());
        } elseif ($result->hasError()) {
            // Invoice was sent to Acumulus but not created due to (an) error(s)
            // in the invoice.
            $this->getAcumulusContainer()->getLog()->debug($logMessage . $result->getMessages(Severity::Error)[0]->getText());
        } else {
            // Sent successfully, invoice has been created in Acumulus:
            if ($result->getMessages(Severity::Warning)) {
                // With warnings.
                $this->getAcumulusContainer()->getLog()->debug($logMessage . $result->getMessages(Severity::Warning)[0]->getText());
            } else {
                // Without warnings.
                $this->getAcumulusContainer()->getLog()->debug($logMessage . 'success');
            }

            // Check if an entry id was created.
            $acumulusInvoice = $result->getMainApiResponse();
            if (!empty($acumulusInvoice['entryid'])) {
                // The invoice has been added.
                $this->getAcumulusContainer()->getLog()->debug($logMessage . 'invoice added');
                $invoiceNumber = $acumulusInvoice['invoicenumber'];
                $entryId = $acumulusInvoice['entryid'];
                $token = $acumulusInvoice['token'];
            } elseif (!empty($acumulusInvoice['conceptid'])) {
                // The invoice has been added as a concept.
                $this->getAcumulusContainer()->getLog()->debug($logMessage . 'concept added');
                $conceptId = $acumulusInvoice['conceptid'];
            } else {
                // The invoice was sent in test modus: no invoice nor a concept has been added.
                $this->getAcumulusContainer()->getLog()->debug($logMessage . 'sent in test modus');
            }
        }
    }
}
