<?php
/**
 * @noinspection AutoloadingIssuesInspection
 * @noinspection PhpIllegalPsrClassPathInspection
 */

declare(strict_types=1);

namespace Opencart\Admin\Controller\Extension\AcumulusCustomiseInvoice\Module;

require_once(DIR_EXTENSION . 'acumulus_customise_invoice/catalog/controller/module/acumulus_customise_invoice.php');

/**
 * This 'admin' controller extends the 'catalog' controller from this extension!
 *
 * The 'catalog' controller contains:
 * - The properties used in 'catalog', and, optionally, here in 'admin'.
 * - A constructor with initialising code.
 * - The event handling methods. The events can be triggered on both the 'admin'
 *   and 'catalog' side. Therefore, they are defined in the 'catalog' controller
 *   and the 'admin' controller inherits them from the 'catalog' controller.
 *
 * The 'admin' controller contains;
 * - Installation code (registering the event handlers).
 * - Uninstall code (removing the registered event handlers).
 *
 * Usage of this extension:
 * - In this part, the 'admin' controller, you should only register the events
 *   you are going to use.
 * - In the other part, the 'catalog' controller, you should change the event
 *   handlers to let them do what you want to achieve and remove the event
 *   handlers you don't need.
 *
 * @noinspection PhpUnused
 */
class AcumulusCustomiseInvoice extends \Opencart\Catalog\Controller\Extension\AcumulusCustomiseInvoice\Module\AcumulusCustomiseInvoice
{
    private static string $name = 'acumulus_customise_invoice';

    /**
     * Install controller action, called when the module is installed.
     *
     * @throws \Exception
     */
    public function install(): void
    {
        $this->installEvents();
    }

    /**
     * Uninstall function, called when the module is uninstalled by an admin.
     *
     * @throws \Exception
     */
    public function uninstall(): void
    {
        $this->uninstallEvents();
    }

    /**
     * Main controller action: show/process the basic settings form for this
     * module.
     */
    public function index(): void
    {
        $name = self::$name;
        $route = "extension/$name/module/$name";
        $this->load->language($route);
        $this->document->setTitle($this->language->get('heading_title'));
        $data['heading_title'] = $this->language->get('heading_title');
        $data['text_form'] = $this->language->get('text_form');

        // Add an intermediate level to the breadcrumb.
        $data['breadcrumbs'] = [];
        $data['breadcrumbs'][] = [
            'text' => $this->language->get('text_home'),
            'href' => $this->url->link('common/dashboard', 'user_token=' . $this->session->data['user_token'], true),
        ];
        $data['breadcrumbs'][] = [
            'text' => $this->language->get('text_extension'),
            'href' => $this->url->link(
                'marketplace/extension',
                'user_token=' . $this->session->data['user_token'] . '&type=module',
                true
            ),
        ];
        $data['breadcrumbs'][] = [
            'text' => $this->language->get('heading_title'),
            'href' => $this->url->link(
                $route,
                'user_token=' . $this->session->data['user_token'],
                true
            ),
        ];

        $data['back'] = $this->url->link('common/dashboard', 'user_token=' . $this->session->data['user_token'], true);

        $data['header'] = $this->load->controller('common/header');
        $data['column_left'] = $this->load->controller('common/column_left');
        $data['footer'] = $this->load->controller('common/footer');

        $this->response->setOutput($this->load->view($route, $data));
    }

    /**
     * Installs our events.
     *
     * This will add them to the table 'event' from where they are registered on
     * the start of each request. The controller actions can be found below.
     *
     * @throws \Exception
     */
    private function installEvents(): void
    {
        $this->uninstallEvents();
        $this->addEvents();
    }

    protected function addEvents(): void
    {
        $this->addEvent('invoiceCreate', 'before');
        $this->addEvent('lineCollect', 'before');
        $this->addEvent('lineCollect', 'after');
        $this->addEvent('invoiceCollect', 'after');
        $this->addEvent('invoiceCreate', 'after');
        $this->addEvent('invoiceSend', 'before');
        $this->addEvent('invoiceSend', 'after');
    }

    protected function addEvent(
        string $trigger,
        string $moment,
        ?string $method = null,
        ?string $code = null,
        bool $status = true,
        int $sort_order = 1
    ):
    void {
        $method ??= $trigger . ucfirst($moment);
        $code ??= self::$name;
        $trigger = $this->getAcumulusOcRegistry()->getAcumulusTrigger($trigger, $moment);
        $action = $this->getAcumulusOcRegistry()->getRoute($method, self::$name);
        $this->model_setting_event->addEvent([
            'code' => $code,
            'description' => '',
            'trigger' => $trigger,
            'action' => $action,
            'status' => $status,
            'sort_order' => $sort_order,
        ]);
    }

    /**
     * Removes the Acumulus event handlers from the event table.
     *
     * @throws \Exception
     */
    private function uninstallEvents(): void
    {
        $this->load->model('setting/event');
        $this->model_setting_event->deleteEventByCode(self::$name);
    }
}
