<?php
/**
 * @author    Buro RaDer, https://burorader.com/
 * @copyright SIEL BV, https://www.siel.nl/acumulus/
 * @license   GPL v3, see license.txt
 *
 * @noinspection AutoloadingIssuesInspection
 * @noinspection PhpUnused Module is instantiated by the PrestaShop runtime, hooks are
 *   called via the event system
 * @noinspection PhpUnusedAliasInspection A number of aliases are added because they are
 *    used in example code or can be expected to be used by custom code that is written on
 *    top of this code.
 */

declare(strict_types=1);

use Siel\Acumulus\Api;
use Siel\Acumulus\Collectors\PropertySources;
use Siel\Acumulus\Data\Invoice;
use Siel\Acumulus\Data\Line;
use Siel\Acumulus\Data\LineType;
use Siel\Acumulus\Data\PropertySet;
use Siel\Acumulus\Fld;
use Siel\Acumulus\Helpers\Container;
use Siel\Acumulus\Helpers\Message;
use Siel\Acumulus\Helpers\Severity;
use Siel\Acumulus\Invoice\InvoiceAddResult;
use Siel\Acumulus\Invoice\Source;
use Siel\Acumulus\Meta;

/**
 * The AcumulusCustomiseInvoice module class contains plumbing and example code
 * to react to hooks triggered by the Acumulus module. These hooks allow you to:
 * - Prevent sending an invoice to Acumulus.
 * - Customise the invoice before, during, or after it is being created.
 * - Process the results of sending the invoice to Acumulus.
 * - Inject custom actions at any of the event moments.
 *
 * Usage of this module:
 *
 * You can use and modify this example module as you like:
 * - only register the hooks you are going to use.
 * - add your own hook handling in those handler methods.
 *
 * Or, if you already have a module with custom code, you can add this code
 * over there:
 * - any hook handling code only copy the hooks you are going to use.
 * - any registerHook() call: only copy those hooks that you need.
 *
 * Documentation for the hooks:
 *
 * The Acumulus module defines the following hooks:
 * 1) {@see \Siel\Acumulus\PrestaShop\Helpers\Event::triggerInvoiceCreateBefore() actionAcumulusInvoiceCreateBefore}
 * 2) {@see \Siel\Acumulus\PrestaShop\Helpers\Event::triggerLineCollectBefore() actionAcumulusLineCollectBefore}
 * 3) {@see \Siel\Acumulus\PrestaShop\Helpers\Event::triggerLineCollectAfter() actionAcumulusLineCollectAfter}
 * 4) {@see \Siel\Acumulus\PrestaShop\Helpers\Event::triggerInvoiceCollectAfter() actionAcumulusInvoiceCollectAfter}
 * 5) {@see \Siel\Acumulus\PrestaShop\Helpers\Event::triggerInvoiceCreateAfter() actionAcumulusInvoiceCreateAfter}
 * 6) {@see \Siel\Acumulus\PrestaShop\Helpers\Event::triggerInvoiceSendBefore() actionAcumulusInvoiceSendBefore}
 * 7) {@see \Siel\Acumulus\PrestaShop\Helpers\Event::triggerInvoiceSendAfter() actionAcumulusInvoiceSendAfter}
 *
 * The hooks are documented in the
 * {@see \Siel\Acumulus\PrestaShop\Helpers\Event Event interface}.
 * The links in the list above point to that documentation, but you probably need
 * generated PHPDoc documentation or a good IDE to follow those links. If not navigable,
 * go to the file {webroot}/modules/acumulus/vendor/siel/acumulus/src/Helpers/Event.php.
 *
 * External Resources:
 *
 * - https://apidoc.sielsystems.nl/content/invoice-add.
 * - https://apidoc.sielsystems.nl/content/warning-error-and-status-response-section-most-api-calls
 */
class Acumulus_Customise_Invoice extends Module
{
    /**
     * Do not call directly, use the getter getAcumulusContainer().
     */
    private Container $acumulusContainer;

    public function __construct()
    {
        /**
         * Increase this value on each change:
         * - point release: bug fixes
         * - minor version: addition of minor features, backwards compatible
         * - major version: major or backwards incompatible changes
         *
         * PrestaShop Note: maximum version length = 8, so do not use alpha or beta.
         */
        $this->version = '8.7.0';
        $this->name = 'acumulus_customise_invoice';
        $this->tab = 'billing_invoicing';
        $this->author = 'Acumulus';
        $this->need_instance = 0;
        $this->ps_versions_compliancy = ['min' => '8', 'max' => '10'];
        $this->dependencies = ['acumulus'];
        $this->bootstrap = true;
        /** @noinspection PhpDynamicFieldDeclarationInspection */
        $this->is_configurable = false;

        parent::__construct();

        $this->displayName = $this->l('Customise Acumulus Invoices');
        $this->description = $this->l('Example module that shows how to alter invoices by reacting to hooks defined by the Acumulus module.');
        $this->confirmUninstall = $this->l('Are you sure you want to uninstall?');
    }

    /**
     * Do not call directly, use getAcumulusContainer().
     */
    private function init(): void
    {
        if (!isset($this->acumulusContainer)) {
            if (Container::getContainer() === null) {
                $languageCode = isset(Context::getContext()->language) ? Context::getContext()->language->iso_code : 'nl';
                $this->acumulusContainer = new Container('PrestaShop', $languageCode);
            } else {
                $this->acumulusContainer = Container::getContainer();
            }
        }
    }

    /**
     * Returns the {@see Container Acumulus container}, so this custom plugin has access
     * to the Acumulus classes, configuration, translations, and constants.
     */
    private function getAcumulusContainer(): Container
    {
        $this->init();
        return $this->acumulusContainer;
    }

    /**
     * Helper method to translate strings (from the Acumulus plugin).
     *
     * @param string $key
     *  The key to get a translation for.
     *
     * @return string
     *   The translation for the given key or the key itself if no translation could be
     *   found.
     */
    private function t(string $key): string
    {
        return $this->getAcumulusContainer()->getTranslator()->get($key);
    }

    /**
     * Activate current module.
     *
     * @param bool $force_all If true, enable module for all shop
     */
    public function enable($force_all = false): bool
    {
        return parent::enable($force_all)
            && $this->registerHook('actionAcumulusInvoiceCreateBefore')
            && $this->registerHook('actionAcumulusLineCollectBefore')
            && $this->registerHook('actionAcumulusLineCollectAfter')
            && $this->registerHook('actionAcumulusInvoiceCollectAfter')
            && $this->registerHook('actionAcumulusInvoiceCreateAfter')
            && $this->registerHook('actionAcumulusInvoiceSendBefore')
            && $this->registerHook('actionAcumulusInvoiceSendAfter');
    }

    /**
     * Desactivate current module.
     *
     * @param bool $force_all If true, disable module for all shop
     */
    public function disable($force_all = false): bool
    {
        $this->unregisterHook('actionAcumulusInvoiceCreateBefore');
        $this->unregisterHook('actionAcumulusLineCollectBefore');
        $this->unregisterHook('actionAcumulusLineCollectAfter');
        $this->unregisterHook('actionAcumulusInvoiceCollectAfter');
        $this->unregisterHook('actionAcumulusInvoiceCreateAfter');
        $this->unregisterHook('actionAcumulusInvoiceSendBefore');
        $this->unregisterHook('actionAcumulusInvoiceSendAfter');
        return parent::disable($force_all);
    }

    /**
     * Processes the hook triggered at the start of the create process.
     *
     * @param array $params
     *   Array with the following entries:
     *   - 'invoiceSource' => {@see \Siel\Acumulus\Invoice\Source}
     *     Wrapper around the original PrestaShop order or refund for which the
     *     invoice will be created.
     *   - 'localResult' => {@see \Siel\Acumulus\Invoice\Result}
     *     Any local error or warning messages that are created locally.
     */
    public function hookActionAcumulusInvoiceCreateBefore(array $params): void
    {
        /** @var \Siel\Acumulus\Invoice\Source $invoiceSource */
        $invoiceSource = $params['invoiceSource'];
        /** @var \Siel\Acumulus\Invoice\InvoiceAddResult $localResult */
        $localResult = $params['localResult'];

        $this->getAcumulusContainer()->getLog()->debug(__METHOD__ . '(order_id=%d)', $invoiceSource->getId());
        try {
            // Do something.
        } catch (Throwable $e) {
            // Prevent creating and sending the invoice:
            $localResult->setSendStatus(InvoiceAddResult::NotSent_LocalErrors);
            $localResult->addMessage(Message::createFromException($e));
            // or just your own error message:
            $localResult->addMessage(Message::create('My message', Severity::Error));
        }
    }

    /**
     * Processes the hook triggered before a line gets collected.
     *
     * @param array $params
     *   Array with the following entries:
     *   - 'line' => {@see \Siel\Acumulus\Data\Line}
     *     The line to be collected.
     *   - 'propertySources' => {@see \Siel\Acumulus\Collectors\PropertySources}
     *     A set of objects that can be used to collect values from.
     */
    public function hookActionAcumulusLineCollectBefore(array $params): void
    {
        /** @var \Siel\Acumulus\Data\Line $line */
        $line = $params['line'];
        /** @var \Siel\Acumulus\Collectors\PropertySources $propertySources */
        $propertySources = $params['propertySources'];

        /** @var \Siel\Acumulus\Invoice\Source $source */
        $source = $propertySources->get('source');
        /** @var \Siel\Acumulus\Data\Invoice $invoice */
        $invoice = $propertySources->get('invoice');

        // Here you can already add values to the line to be collected or add or change
        // some propertySources.
        $this->getAcumulusContainer()->getLog()->debug(__METHOD__ . '(type=%s, order-id=%d)', $line->getType(), $source->getId());
        $someValue = random_int(1, 100);
        $propertySources->add('myObject', $someValue);
        $someCondition = $someValue === 0;
        if ($someCondition) {
            // Do not add this line.
            $line->metadataSet(Meta::DoNotAdd, true);
        }
    }

    /**
     * Processes the hook triggered after a line got collected.
     *
     * @param array $params
     *   Array with the following entries:
     *   - 'line' => {@see \Siel\Acumulus\Data\Line}
     *     The line to be collected.
     *   - 'propertySources' => {@see \Siel\Acumulus\Collectors\PropertySources}
     *     A set of objects that can be used to collect values from.
     */
    public function hookActionAcumulusLineCollectAfter(array $params): void
    {
        /** @var \Siel\Acumulus\Data\Line $line */
        $line = $params['line'];
        /** @var \Siel\Acumulus\Collectors\PropertySources $propertySources */
        $propertySources = $params['propertySources'];

        /** @var \Siel\Acumulus\Invoice\Source $source */
        $source = $propertySources->get('source');
        /** @var \Siel\Acumulus\Data\Invoice $invoice */
        $invoice = $propertySources->get('invoice');

        // Here you can make changes to the collected line based on your situation,
        // as well as remove added propertySources or prevent adding this line after all.
        $this->getAcumulusContainer()->getLog()->debug(__METHOD__ . '(type=%s, order-id=%d)', $line->getType(), $source->getId());
        $propertySources->remove('myObject');
    }

    /**
     * Processes the hook triggered after the raw invoice has been collected
     * but before it will be completed.
     *
     * Note that the invoice needs yet to be completed:
     * - Fields that depend on configuration are yet to be filled in.
     * - Child lines are still hierarchical.
     * - Foreign amounts are not yet converted to euros.
     * - Amounts may be missing, e.g. only  unitpriceinc' and 'vatamount' are
     *   filled in, while unitprice and vatrate are not yet computed.
     *
     * @param array $params
     *   Array with the following entries:
     *   - 'invoice' => {@see \Siel\Acumulus\Invoice\Invoice}
     *     The invoice in Acumulus format as will be completed and sent to
     *     Acumulus.
     *   - 'invoiceSource' => {@see \Siel\Acumulus\Invoice\Source}
     *     Wrapper around the original PrestaShop order or refund for which the
     *     invoice has been collected.
     *   - 'localResult' => {@see \Siel\Acumulus\Invoice\Result}
     *     Any error or warning messages that were created locally.
     */
    public function hookActionAcumulusInvoiceCollectAfter(array $params): void
    {
        /** @var \Siel\Acumulus\Data\Invoice $invoice */
        $invoice = $params['invoice'];
        /** @var \Siel\Acumulus\Invoice\Source $invoiceSource */
        $invoiceSource = $params['invoiceSource'];
        /** @var \Siel\Acumulus\Invoice\InvoiceAddResult $localResult */
        $localResult = $params['localResult'];

        // Here you can make changes to the raw invoice based on your situation.
        $this->getAcumulusContainer()->getLog()->debug(__METHOD__ . '(order_id=%d)', $invoiceSource->getId());
    }

    /**
     * Processes the hook triggered after the invoice has been created.
     *
     * @param array $params
     *   Array with the following entries:
     *   - 'invoice' => {@see \Siel\Acumulus\Invoice\Invoice}
     *     The invoice in Acumulus format as will be sent to Acumulus.
     *   - 'invoiceSource' => {@see \Siel\Acumulus\Invoice\Source}
     *     Wrapper around the original PrestaShop order or refund for which the
     *     invoice has been created.
     *   - 'localResult' => {@see \Siel\Acumulus\Invoice\Result}
     *     Any error or warning messages that were created locally.
     */
    public function hookActionAcumulusInvoiceCreateAfter(array $params): void
    {
        /** @var \Siel\Acumulus\Data\Invoice $invoice */
        $invoice = $params['invoice'];
        /** @var \Siel\Acumulus\Invoice\Source $invoiceSource */
        $invoiceSource = $params['invoiceSource'];
        /** @var \Siel\Acumulus\Invoice\InvoiceAddResult $localResult */
        $localResult = $params['localResult'];

        // Here you can make changes to the invoice based on your situation.
        $this->getAcumulusContainer()->getLog()->debug(__METHOD__ . '(order_id=%d)', $invoiceSource->getId());

        // Real exmple: setting or changing the account number or template before the
        // invoice will be sent to Acumulus.
        if ($invoiceSource->getType() === Source::Order) {
            /** @var \Order $order */
            $order = $invoiceSource->getShopObject();
            if ($order->module === 'mollie') {
                // Refine account based on payment method, not only on payment module as
                // PrestaShop payment modules may be all-in-one modules.
                if (str_contains(strtolower($order->payment), 'paypal')) {
                    $acumulusApiClient = $this->getAcumulusContainer()->getAcumulusApiClient();
                    $accountNumberList = $acumulusApiClient->getPicklistAccounts()->getMainAcumulusResponse();
                    $accountNumber = null;
                    foreach ($accountNumberList as $account) {
                        if (str_contains(strtolower($account['accountdescription']), 'paypal')) {
                            $accountNumber = $account['accountnumber'];
                            break;
                        }
                    }
                    $invoice->setAccountNumber($accountNumber, PropertySet::NotEmpty);
                }
            }
        }
    }

    /**
     * Processes the hook triggered after a line got collected.
     *
     * @param array $params
     *   Array with the following entries:
     *   - 'invoice' => {@see \Siel\Acumulus\Invoice\Invoice}
     *     The invoice in Acumulus format as will be sent to Acumulus.
     *   - 'localResult' => {@see \Siel\Acumulus\Invoice\Result}
     *     Any local error or warning messages that were created locally.
     */
    public function hookActionAcumulusInvoiceSendBefore(array $params): void
    {
        /** @var \Siel\Acumulus\Data\Invoice $invoice */
        $invoice = $params['invoice'];
        /** @var \Siel\Acumulus\Invoice\InvoiceAddResult $localResult */
        $localResult = $params['localResult'];

        // Here you can make changes to the completed invoice based on your situation,
        $this->getAcumulusContainer()->getLog()->debug(__METHOD__ . '(order_id=%d)', $invoice->metadataGet(Meta::SourceId));
    }

    /**
     * Processes the hook triggered after an invoice has been sent to Acumulus.
     *
     * @param array $params
     *   Array with the following entries:
     *   - 'invoice' => {@see \Siel\Acumulus\Invoice\Invoice}
     *     The invoice in Acumulus format as will be sent to Acumulus.
     *   - 'invoiceSource' => {@see \Siel\Acumulus\Invoice\Source}
     *     Wrapper around the original PrestaShop order or refund for which the
     *     invoice has been created.
     *   - 'result' => {@see \Siel\Acumulus\Invoice\Result}
     *     The result as sent back by Acumulus + any local messages and warnings.
     */
    public function hookActionAcumulusInvoiceSendAfter(array $params): void
    {
        /** @var \Siel\Acumulus\Data\Invoice $invoice */
        $invoice = $params['invoice'];
        /** @var \Siel\Acumulus\Invoice\Source $invoiceSource */
        $invoiceSource = $params['invoiceSource'];
        /** @var \Siel\Acumulus\Invoice\InvoiceAddResult $result */
        $result = $params['result'];

        // Here you can react to the result of sending the invoice to Acumulus
        $logMessage = sprintf(__METHOD__ . '(order_id=%d): ', $invoiceSource->getId());
        if ($result->getMessages(Severity::Exception)) {
            $this->getAcumulusContainer()->getLog()->debug($logMessage . $result->getMessages(Severity::Exception)[0]->getText());
        } elseif ($result->hasError()) {
            // Invoice was sent to Acumulus but not created due to (an) error(s)
            // in the invoice.
            $this->getAcumulusContainer()->getLog()->debug($logMessage . $result->getMessages(Severity::Error)[0]->getText());
        } else {
            // Sent successfully, invoice has been created in Acumulus:
            if ($result->getMessages(Severity::Warning)) {
                // With warnings.
                $this->getAcumulusContainer()->getLog()->debug($logMessage . $result->getMessages(Severity::Warning)[0]->getText());
            } else {
                // Without warnings.
                $this->getAcumulusContainer()->getLog()->debug($logMessage . 'success');
            }

            // Check if an entry id was created.
            $acumulusInvoice = $result->getMainApiResponse();
            if (!empty($acumulusInvoice['entryid'])) {
                // The invoice has been added.
                $this->getAcumulusContainer()->getLog()->debug($logMessage . 'invoice added');
                $invoiceNumber = $acumulusInvoice['invoicenumber'];
                $entryId = $acumulusInvoice['entryid'];
                $token = $acumulusInvoice['token'];
            } elseif (!empty($acumulusInvoice['conceptid'])) {
                // The invoice has been added as a concept.
                $this->getAcumulusContainer()->getLog()->debug($logMessage . 'concept added');
                $conceptId = $acumulusInvoice['conceptid'];
            } else {
                // The invoice was sent in test modus: no invoice nor a concept has been added.
                $this->getAcumulusContainer()->getLog()->debug($logMessage . 'sent in test modus');
            }
        }
    }
}
