<?php

namespace Siel\Acumulus\Shop;

use DateTime;
use Siel\Acumulus\Api;
use Siel\Acumulus\ApiClient\Acumulus;
use Siel\Acumulus\Config\Config;
use Siel\Acumulus\Config\ShopCapabilities;
use Siel\Acumulus\Helpers\Form;
use Siel\Acumulus\Helpers\FormHelper;
use Siel\Acumulus\Helpers\Log;
use Siel\Acumulus\Helpers\Severity;
use Siel\Acumulus\Helpers\Translator;
use Siel\Acumulus\Tag;

/**
 * Class RegisterForm implements a registration form to register for a
 * temporary free Acumulus account (which can be converted to a full account).
 * It is similar to the form on https://www.siel.nl/acumulus/proefaccount/,
 * though via the API we will also get authentication details for an API
 * account.
 *
 * @noinspection PhpUnused Instantiated by \Siel\Acumulus\Helpers\Container::getForm().
 */
class RegisterForm extends Form
{
    /**
     * @var array
     *   The response structure of a successful sign-up call,
     *   {@see https://www.siel.nl/acumulus/API/Sign_Up/Sign_Up/} for more
     *   details.
     */
    protected $signUpResponse;

    /**
     * RegisterForm constructor.
     *
     * @param \Siel\Acumulus\ApiClient\Acumulus $acumulusApiClient
     * @param \Siel\Acumulus\Helpers\FormHelper $formHelper
     * @param \Siel\Acumulus\Config\ShopCapabilities $shopCapabilities
     * @param \Siel\Acumulus\Config\Config $config
     * @param \Siel\Acumulus\Helpers\Translator $translator
     * @param \Siel\Acumulus\Helpers\Log $log
     */
    public function __construct(Acumulus $acumulusApiClient, FormHelper $formHelper, ShopCapabilities $shopCapabilities, Config $config, Translator $translator, Log $log)
    {
        parent::__construct($acumulusApiClient, $formHelper, $shopCapabilities, $config, $translator, $log);
        $this->translator->add(new RegisterFormTranslations());
        $this->signUpResponse = null;
    }

    /**
     * @inheritDoc
     */
    protected function validate()
    {
        $regexpEmail = '/^[^@<>,; "\']+@([^.@ ,;]+\.)+[^.@ ,;]+$/';

        if (empty($this->submittedValues[Tag::Gender])) {
            $this->addMessage(sprintf($this->t('message_validate_required_field'), $this->t('field_gender')), Severity::Error, Tag::Gender);
        }

        if (empty($this->submittedValues[Tag::FullName])) {
            $this->addMessage(sprintf($this->t('message_validate_required_field'), $this->t('field_fullName')), Severity::Error, Tag::FullName);
        }

        if (empty($this->submittedValues[Tag::LoginName])) {
            $this->addMessage(sprintf($this->t('message_validate_required_field'), $this->t('field_loginName')), Severity::Error, Tag::LoginName);
        } elseif (mb_strlen($this->submittedValues[Tag::LoginName]) < 6) {
            $this->addMessage(sprintf($this->t('message_validate_loginname_0'), $this->t('field_loginName')), Severity::Error, Tag::LoginName);
        }

        if (empty($this->submittedValues[Tag::CompanyTypeId]) && $this->submittedValues[Tag::CompanyTypeId] !== 0) {
            $this->addMessage(sprintf($this->t('message_validate_required_field'), $this->t('field_companyTypeId')), Severity::Error, Tag::CompanyTypeId);
        }

        if (empty($this->submittedValues[Tag::CompanyName])) {
            $this->addMessage(sprintf($this->t('message_validate_required_field'), $this->t('field_companyName')), Severity::Error, Tag::CompanyName);
        }

        if (empty($this->submittedValues[Tag::Address])) {
            $this->addMessage(sprintf($this->t('message_validate_required_field'), $this->t('field_address')), Severity::Error, Tag::Address);
        }

        if (empty($this->submittedValues[Tag::PostalCode])) {
            $this->addMessage(sprintf($this->t('message_validate_required_field'), $this->t('field_postalCode')), Severity::Error, Tag::PostalCode);
        } elseif (!preg_match('/^\d{4}\s*[a-zA-Z]{2}$/', $this->submittedValues[Tag::PostalCode])) {
            $this->addMessage($this->t('message_validate_postalCode_0'), Severity::Error, Tag::PostalCode);
        }

        if (empty($this->submittedValues[Tag::City])) {
            $this->addMessage(sprintf($this->t('message_validate_required_field'), $this->t('field_city')), Severity::Error, Tag::City);
        }

        if (empty($this->submittedValues[Tag::Email])) {
            $this->addMessage(sprintf($this->t('message_validate_required_field'), $this->t('field_email')), Severity::Error, Tag::Email);
        } elseif (!preg_match($regexpEmail, $this->submittedValues[Tag::Email])) {
            $this->addMessage($this->t('message_validate_email_0'), Severity::Error, Tag::Email);
        }
    }

    /**
     * @inheritDoc
     */
    protected function execute()
    {
        $tags = [
            Tag::CompanyTypeId,
            Tag::CompanyName,
            Tag::FullName,
            Tag::LoginName,
            Tag::Gender,
            Tag::Address,
            Tag::PostalCode,
            Tag::City,
            Tag::Email,
            Tag::Telephone,
            Tag::BankAccount,
            Tag::Notes,
        ];
        $submittedValues = $this->submittedValues;
        $signUp = [];
        foreach ($tags as $tag) {
            $this->addIfIsset($signUp, $tag, $submittedValues);
        }

        // Complete $signUp with non-form values.
        if (empty($signUp[Tag::Gender])) {
            $signUp[Tag::Gender] = Api::Gender_Neutral;
        }
        $signUp[Tag::CreateApiUser] = Api::CreateApiUser_Yes;

        $result = $this->acumulusApiClient->signUp($signUp);

        $this->addMessages($result->getMessages(Severity::WarningOrWorse));
        $formSuccess = !$result->hasError();
        if ($formSuccess) {
            $this->signUpResponse = $result->getResponse();
            $this->setAccountInfo($this->signUpResponse);
        }

        return $formSuccess;
    }

    /**
     * Processes the account info as received from Acumulus
     *
     * @param array $signUpResponse
     *   The new account info. A keyed array with the keys:
     *   - 'contractcode'
     *   - 'contractloginname'
     *   - 'contractpassword'
     *   - 'contractstartdate'
     *   - 'contractenddate'
     *   - 'contractapiuserloginname'
     *   - 'contractapiuserpassword'
     *   {@see https://www.siel.nl/acumulus/API/Sign_Up/Sign_Up/} for more
     *   details.
     *
     * @return bool
     *   True on success, false on failure.
     */
    protected function setAccountInfo(array $signUpResponse)
    {
        $accountValues = [
            Tag::ContractCode => $signUpResponse[Tag::ContractCode],
            Tag::UserName => $signUpResponse['contractapiuserloginname'],
            Tag::Password => $signUpResponse['contractapiuserpassword'],
            Tag::EmailOnError => $this->getSubmittedValue(Tag::Email),
        ];
        return $this->acumulusConfig->save($accountValues);
    }

    /**
     * @inheritDoc
     */
    protected function getFieldDefinitions()
    {
        // Test success screen
//        $this->signUpResponse = [
//            'contractcode' => '218975',
//            'contractloginname' => 'erwind',
//            'contractpassword' => 'WCpAfaW8hABq',
//            'contractstartdate' => '2020-05-25',
//            'contractenddate' => '2020-06-24',
//            'contractapiuserloginname' => 'Acumulus-API-ce5bb',
//            'contractapiuserpassword' => 'OTAcu3eq5VjezM',
//        ];
//        $this->submittedValues[Tag::Email] = 'erwin@burorader.com';
        // End test success screen

        $fields = [];
        if ($this->signUpResponse === null) {
            // Not submitted or errors: render registration form.
            $fields += [
                'introHeader' => [
                    'type' => 'fieldset',
                    'legend' => $this->t('introHeader'),
                    'fields' => $this->getIntroFields(),
                ],
                'personSettingsHeader' => [
                    'type' => 'fieldset',
                    'legend' => $this->t('personSettingsHeader'),
                    'fields' => $this->getPersonFields(),
                ],
                'companySettingsHeader' => [
                    'type' => 'fieldset',
                    'legend' => $this->t('companySettingsHeader'),
                    'description' => $this->t('desc_companySettings'),
                    'fields' => $this->getCompanyFields(),
                ],
                'notesSettingsHeader' => [
                    'type' => 'fieldset',
                    'legend' => $this->t('notesSettingsHeader'),
                    'fields' => $this->getNotesFields(),
                ],
            ];
        } else {
            // Successfully submitted: show details of the created account.
            $this->loadInfoBlockTranslations();
            $this->needsFormAndSubmitButton = false;
            $fields += $this->getCreatedAccountFields();
            $fields += $this->getCreatedApiAccountFields();
            $fields += $this->getNextSteps();
            $fields['versionInformationHeader'] = $this->getInformationBlock();
        }
        return $fields;
    }

    /**
     * Returns the set of intro fields.
     *
     * The fields returned:
     * - intro
     *
     * @return array[]
     *   The set of intro fields.
     */
    protected function getIntroFields()
    {
        return [
            'intro' => [
                'type' => 'markup',
                'label' => $this->getLogo(),
                'value' => $this->t('register_form_intro'),
                'attributes' => [
                    'label' => [
                        'html' => true,
                    ],
                ],
            ],
        ];
    }

    /**
     * Returns the set of personal related fields.
     *
     * The fields returned:
     * - gender
     * - fullname
     * - loginname
     * - email
     * - telephone
     *
     * @return array[]
     *   The set of personal related fields.
     */
    protected function getPersonFields()
    {
        return [
            Tag::Gender => [
                'type' => 'radio',
                'label' => $this->t('field_gender'),
                'description' => $this->t('desc_gender'),
                'options' => [
                    Api::Gender_Neutral => $this->t('option_gender_neutral'),
                    Api::Gender_Female => $this->t('option_gender_female'),
                    Api::Gender_Male => $this->t('option_gender_male'),
                ],
                'attributes' => [
                    'required' => true,
                ],
            ],
            Tag::FullName => [
                'type' => 'text',
                'label' => $this->t('field_fullName'),
                'description' => $this->t('desc_fullName'),
                'attributes' => [
                    'required' => true,
                    'size' => 40,
                ],
            ],
            Tag::LoginName => [
                'type' => 'text',
                'label' => $this->t('field_loginName'),
                'description' => sprintf($this->t('desc_loginName'), $this->t('module')),
                'attributes' => [
                    'required' => true,
                    'size' => 20,
                ],
            ],
        ];
    }

    /**
     * Returns the set of company related fields.
     *
     * The fields returned:
     * - companyname
     * - address
     * - postalcode
     * - city
     * - bankaccount
     *
     * @return array[]
     *   The set of notes related fields.
     */
    protected function getCompanyFields()
    {
        return [
            Tag::CompanyTypeId => [
                'type' => 'select',
                'label' => $this->t('field_companyTypeId'),
                'options' => $this->picklistToOptions($this->acumulusApiClient->getPicklistCompanyTypes(), '', $this->t('option_empty')),
                'attributes' => [
                    'required' => true,
                ],
            ],
            Tag::CompanyName => [
                'type' => 'text',
                'label' => $this->t('field_companyName'),
                'attributes' => [
                    'required' => true,
                    'size' => 40,
                ],
            ],
            Tag::Address => [
                'type' => 'text',
                'label' => $this->t('field_address'),
                'attributes' => [
                    'required' => true,
                    'size' => 40,
                ],
            ],
            Tag::PostalCode => [
                'type' => 'text',
                'label' => $this->t('field_postalCode'),
                'attributes' => [
                    'required' => true,
                    'size' => 8,
                ],
            ],
            Tag::City => [
                'type' => 'text',
                'label' => $this->t('field_city'),
                'attributes' => [
                    'required' => true,
                    'size' => 20,
                ],
            ],
            Tag::Email => [
                'type' => 'email',
                'label' => $this->t('field_emailRegistration'),
                'description' => sprintf($this->t('desc_emailRegistration'), $this->t('module')),
                'attributes' => [
                    'required' => true,
                    'size' => 40,
                ],
            ],
            Tag::Telephone => [
                'type' => 'text',
                'label' => $this->t('field_telephone'),
                'description' => $this->t('desc_telephone'),
                'attributes' => [
                    'size' => 12,
                ],
            ],
            Tag::BankAccount => [
                'type' => 'text',
                'label' => $this->t('field_bankAccount'),
                'description' => $this->t('desc_bankAccount'),
                'attributes' => [
                    'size' => 20,
                ],
            ],
        ];
    }

    /**
     * Returns the set of notes related fields.
     *
     * The fields returned:
     * - notes
     *
     * @return array[]
     *   The set of notes related fields.
     */
    protected function getNotesFields()
    {
        return [
            Tag::Notes => [
                'type' => 'textarea',
                'label' => $this->t('field_notes'),
                'description' => sprintf($this->t('desc_notes'), $this->t('module')),
                'attributes' => [
                    'rows' => 6,
                ],
            ],
        ];
    }

    /**
     * Returns explanatory text about the test account that has been created.
     *
     * @return array[]
     *   Markup that gives more information about the the test account that has
     *   been created.
     */
    protected function getCreatedAccountFields()
    {
        $title = $this->t('register_form_success_title');
        $line1 = sprintf($this->t('register_form_success_text1'), DateTime::createFromFormat(API::DateFormat_Iso, $this->signUpResponse['contractenddate'])->format('d-m-Y'));
        $line2 = sprintf($this->t('register_form_success_text2'), htmlspecialchars($this->getSubmittedValue(Tag::Email), ENT_NOQUOTES | ENT_HTML5, 'UTF-8'));
        $line3 = $this->t('register_form_success_text3');
        return [
            'congratulations' => [
                'type' => 'markup',
                'value' => "<h1>$title</h1>\n<p>$line1</p>\n<p>$line2 $line3</p>",
            ],
            'loginDetails' => [
                'type' => 'details',
                'summary' => $this->t('loginDetailsHeader'),
                'fields' => [
                    Tag::ContractCode => [
                        'type' => 'text',
                        'label' => $this->t('field_code'),
                        'attributes' => [
                            'readonly' => true,
                            'size' => 8,
                        ],
                        'value' => $this->signUpResponse[Tag::ContractCode],
                    ],
                    Tag::LoginName => [
                        'type' => 'text',
                        'label' => $this->t('field_loginName'),
                        'attributes' => [
                            'readonly' => true,
                            'size' => 20,
                        ],
                        'value' => $this->signUpResponse['contract' . Tag::LoginName],
                    ],
                    Tag::Password => [
                        'type' => 'text',
                        'label' => $this->t('field_password'),
                        'attributes' => [
                            'readonly' => true,
                            'size' => 20,
                        ],
                        'value' => $this->signUpResponse['contract' . Tag::Password],
                    ],
                ],
            ],
        ];
    }

    /**
     * Returns explanatory text about the test account that has been created.
     *
     * @return array[]
     *   Markup that gives more information about the the test account that has
     *   been created.
     */
    protected function getCreatedApiAccountFields()
    {
        $line1 = sprintf($this->t('register_form_success_api_account'), $this->t('module'));
        return [
            'apiAccount' => [
                'type' => 'markup',
                'value' => "<p>$line1</p>",
            ],
            'apiloginDetails' => [
                'type' => 'details',
                'summary' => sprintf($this->t('moduleLoginDetailsHeader'), $this->t('module')),
                'fields' => [
                    'contractapiuser' . Tag::ContractCode => [
                        'type' => 'text',
                        'label' => $this->t('field_code'),
                        'attributes' => [
                            'readonly' => true,
                            'size' => 8,
                        ],
                        'value' => $this->signUpResponse[Tag::ContractCode],
                    ],
                    'contractapiuser' . Tag::LoginName => [
                        'type' => 'text',
                        'label' => $this->t('field_loginName'),
                        'attributes' => [
                            'readonly' => true,
                            'size' => 20,
                        ],
                        'value' => $this->signUpResponse['contractapiuser' . Tag::LoginName],
                    ],
                    'contractapiuser' . Tag::Password => [
                        'type' => 'text',
                        'label' => $this->t('field_password'),
                        'attributes' => [
                            'readonly' => true,
                            'size' => 20,
                        ],
                        'value' => $this->signUpResponse['contractapiuser' . Tag::Password],
                    ],
                    'desc_apiloginDetails' => [
                        'type' => 'markup',
                        'value' => sprintf($this->t('desc_apiloginDetails'), $this->t('module')),
                    ],
                ],
            ],
        ];
    }

    /**
     * Returns explanatory text about what to do next.
     *
     * @return array[]
     *   Markup that explains what to do next.
     */
    protected function getNextSteps()
    {
        $title = $this->t('whatsNextHeader');
        $line1 = $this->t('register_form_success_configure_acumulus');
        $button1 = $this->t('register_form_success_login_button');
        $line2 = sprintf($this->t('register_form_success_configure_module'), $this->t('module'));
        $button2 = sprintf($this->t('register_form_success_config_button'), $this->t('module'), $this->shopCapabilities->getLink('config'));
        $line3 = sprintf($this->t('register_form_success_batch'), $this->t('module'));
        return [
            'nextSteps' => [
                'type' => 'markup',
                'value' => "<h2>$title</h2><p>$line1</p>\n<p>$button1<br><br></p>\n<p>$line2</p>\n<p>$button2<br><br></p>\n<p>$line3</p>\n",
            ],
        ];
    }
}
