<?php
/*
 * Plugin Name: Acumulus Customise Invoice
 * Description: Plugin to customise Acumulus invoices before sending them
 * Author: Buro RaDer, https://burorader.com/
 * Copyright: SIEL BV, https://www.siel.nl/acumulus/
 * Version: 8.6.6
 * LICENCE: GPLv3
 * Requires at least: 5.9
 * Tested up to: 6.8
 * WC requires at least: 9.1.0
 * WC tested up to: 10
 * Requires PHP: 8.1
 *
 * Remarks about "WC Requires at least":
 * - Stock handling:
 *   - restock: exists since 9.1.0: do_action('woocommerce_restore_order_item_stock', ...);
 *   - decrease stock: exists since 7.6.0: do_action('woocommerce_reduce_order_item_stock', ...);
 * - If stock handling is not used: 5.0 should work.
*/

/**
 * @noinspection AutoloadingIssuesInspection
 * @noinspection PhpUnused
 */

declare(strict_types=1);

use Automattic\WooCommerce\Utilities\FeaturesUtil;
use Siel\Acumulus\Api;
use Siel\Acumulus\Collectors\PropertySources;
use Siel\Acumulus\Data\Invoice;
use Siel\Acumulus\Data\Line;
use Siel\Acumulus\Data\LineType;
use Siel\Acumulus\Fld;
use Siel\Acumulus\Helpers\Container;
use Siel\Acumulus\Helpers\Message;
use Siel\Acumulus\Helpers\Severity;
use Siel\Acumulus\Invoice\InvoiceAddResult;
use Siel\Acumulus\Invoice\Source;
use Siel\Acumulus\Meta;

/**
 * Example code that reacts on Acumulus or WooCommerce actions and filters.
 *
 * NOTE: This code is not ready to run.
 *
 * The hooks reacting to Acumulus events do not really anything other than some logging.
 * You will have to replace it with your own logic in those hooks you want to use. For the
 * hooks you do not use, you should comment-out the add_action() call in the
 * {@see \AcumulusCustomiseInvoice::bootstrap()} method that "activates" them.
 * Note that all hooks are actions. As objects are passed you can easily make changes to
 * these objects without having to return them in the return value. This also holds when
 * you want to prevent sending the invoice: you do so by setting the send status as is
 * done in the example code in {@see \AcumulusCustomiseInvoice::invoiceCreateBefore()}.
 *
 * The code in {@see \AcumulusCustomiseInvoice::wooCommerceMyAccountMyOrdersActions()} is
 * code that is working and that you can use to present Acumulus PDF invoices to your
 * clients in their "my orders" page. If you want to use this feature, please uncomment
 * the <code>add_filter('woocommerce_my_account_my_orders_actions', ...);</code> line in
 * the {@see \AcumulusCustomiseInvoice::bootstrap()} method.
 *
 * Documentation for the events:
 *
 * The Acumulus module defines the following events:
 * 1) {@see \Siel\Acumulus\Helpers\Event::triggerInvoiceCreateBefore() invoiceCreateBefore}
 * 2) {@see \Siel\Acumulus\Helpers\Event::triggerLineCollectBefore() lineCollectBefore}
 * 3) {@see \Siel\Acumulus\Helpers\Event::triggerLineCollectAfter() lineCollectAfter}
 * 4) {@see \Siel\Acumulus\Helpers\Event::triggerInvoiceCollectAfter() invoiceCollectAfter}
 * 5) {@see \Siel\Acumulus\Helpers\Event::triggerInvoiceCreateAfter() invoiceCreateAfter}
 * 6) {@see \Siel\Acumulus\Helpers\Event::triggerInvoiceSendBefore() invoiceSendBefore}
 * 7) {@see \Siel\Acumulus\Helpers\Event::triggerInvoiceSendAfter() invoiceSendAfter}
 *
 * These events are documented in the
 * {@see \Siel\Acumulus\PrestaShop\Helpers\Event Event interface}.
 * The links in the list above point to that documentation, but you probably need
 * generated PHPDoc documentation or a good IDE to see them as links and be able to follow
 * those links. If not navigable, go to the file
 * {webroot}/modules/acumulus/vendor/siel/acumulus/src/Helpers/Event.php.
 *
 * External Resources:
 *
 * - https://apidoc.sielsystems.nl/content/invoice-add.
 * - https://apidoc.sielsystems.nl/content/warning-error-and-status-response-section-most-api-calls
 *
 * Examples:
 *
 * The documentation mentioned above and methods (event handlers) below contain some
 * example code to give yu an idea of what you can do, what information is available, etc.
 */
class AcumulusCustomiseInvoice
{
    private static AcumulusCustomiseInvoice $instance;
    /**
     * Do not call directly, use the getter getAcumulusContainer().
     */
    private Container $container;

    /**
     * Entry point for our plugin.
     */
    public static function create(): AcumulusCustomiseInvoice
    {
        if (!isset(self::$instance)) {
            self::$instance = new self();
        }
        return self::$instance;
    }

    /**
     * Private constructor, to ensure there will be only 1 instance.
     */
    private function __construct()
    {
    }

    /**
     * Bootstraps the environment for the plugin.
     */
    public function bootstrap(): void
    {
        // Install/uninstall actions.
        $file = str_replace('\\', '/', __FILE__);
        register_activation_hook($file, [$this, 'activate']);
        register_deactivation_hook($file, [$this, 'deactivate']);
        register_uninstall_hook($file, ['Acumulus', 'uninstall']);

        // WooCommerce HPOS compatibility. Declare compatibility.
        add_action('before_woocommerce_init', static function () {
            if (class_exists(FeaturesUtil::class)) {
                FeaturesUtil::declare_compatibility('custom_order_tables', __FILE__, true);
            }
        });

        // Actions.
        // Uncomment when you want to show links to the Acumulus invoices on the
        // "My Account - My orders" list.
        add_filter('woocommerce_my_account_my_orders_actions', [$this, 'wooCommerceMyAccountMyOrdersActions'], 10, 2);

        add_action('acumulus_invoice_create_before', [$this, 'invoiceCreateBefore'], 10, 2);
        add_action('acumulus_line_collect_before', [$this, 'lineCollectBefore'], 10, 2);
        add_action('acumulus_line_collect_after', [$this, 'lineCollectAfter'], 10, 2);
        add_action('acumulus_invoice_collect_after', [$this, 'invoiceCollectAfter'], 10, 3);
        add_action('acumulus_invoice_create_after', [$this, 'invoiceCreateAfter'], 10, 3);
        add_action('acumulus_invoice_send_before', [$this, 'invoiceSendBefore'], 10, 2);
        add_action('acumulus_invoice_send_after', [$this, 'invoiceSendAfter'], 10, 3);
    }

    /**
     * Returns the {@see Container Acumulus container}, so this custom plugin has access
     * to the Acumulus classes, configuration, translations, and constants.
     */
    private function getAcumulusContainer(): Container
    {
        if (!isset($this->container)) {
            $this->container = Acumulus::create()->getAcumulusContainer();
        }
        return $this->container;
    }

    /**
     * Helper method to translate strings (from the Acumulus plugin).
     *
     * @param string $key
     *  The key to get a translation for.
     *
     * @return string
     *   The translation for the given key or the key itself if no translation could be
     *   found.
     */
    private function t(string $key): string
    {
        return $this->getAcumulusContainer()->getTranslator()->get($key);
    }

    /**
     * Add any activate code as needed.
     *
     * @return bool
     */
    public function activate(): bool
    {
        if (!current_user_can('activate_plugins')) {
            return false;
        }
        // Optionally add any activate code here.
        return true;
    }

    /**
     * Add any deactivate code as needed.
     *
     * @return bool
     */
    public function deactivate(): bool
    {
        if (!current_user_can('activate_plugins')) {
            return false;
        }
        // Optionally add any deactivate code here.
        return true;
    }

    /**
     * Add any uninstall code as needed.
     *
     * @return bool
     */
    public static function uninstall(): bool
    {
        if (!current_user_can('delete_plugins')) {
            return false;
        }
        // Optionally add any uninstall code here.
        return true;
    }

    /**
     * Shows links to theAcumulus invoices on the My Account - My Orders list
     *
     * Styling to be done via css in your (child) theme.
     *
     * @param array $actions
     *   The actions list defined so far, can be changed/extended here.
     * @param \WC_Abstract_Order $order
     *   The order for which to add actions.
     *
     * @return array
     *   The modified actions list.
     */
    public function wooCommerceMyAccountMyOrdersActions(array $actions, WC_Abstract_Order $order): array
    {
        $source = $this->getAcumulusContainer()->createSource(Source::Order, $order);
        $entry = $this->getAcumulusContainer()->getAcumulusEntryManager()->getByInvoiceSource($source);
        if ($entry !== null) {
            $token = $entry->getToken();
            if ($token !== null) {
                $url = $this->getAcumulusContainer()->getAcumulusApiClient()->getInvoicePdfUri($token);
                $actions['acumulus-invoice'] = ['url' => $url, 'name' => $this->t('Invoice')];
            }
        }
        return $actions;
    }

    /**
     * Processes the event triggered before the start of the create process.
     */
    public function invoiceCreateBefore(Source $invoiceSource, InvoiceAddResult $localResult): void
    {
        $this->getAcumulusContainer()->getLog()->debug(__METHOD__ . '(order_id=%d)', $invoiceSource->getId());
        try {
            // Do something.
        } catch (Throwable $e) {
            // Prevent creating and sending the invoice:
            $localResult->setSendStatus(InvoiceAddResult::NotSent_LocalErrors);
            $localResult->addMessage(Message::createFromException($e));
            // or just your own error message:
            $localResult->addMessage(Message::create('My message', Severity::Error));
        }
    }

    /**
     * Processes the event triggered before a line gets collected.
     *
     * This event will be called once for each line regardless its line type.
     * Line type being one of the {@see \Siel\Acumulus\Data\LineType} constants.
     */
    public function lineCollectBefore(Line $line, PropertySources $propertySources): void
    {
        /** @var \Siel\Acumulus\Invoice\Source $source */
        $source = $propertySources->get('source');
        /** @var \Siel\Acumulus\Data\Invoice $invoice */
        $invoice = $propertySources->get('invoice');

        // Here you can already add values to the line to be collected or add or change
        // some propertySources.
        $this->getAcumulusContainer()->getLog()->debug(__METHOD__ . '(type=%s, order-id=%d)', $line->getType(), $source->getId());
        $someValue = random_int(1, 100);
        $propertySources->add('myObject', $someValue);
        $someCondition = $someValue === 0;
        if ($someCondition) {
            // Do not add this line.
            $line->metadataSet(Meta::DoNotAdd, true);
        }
    }

    /**
     * Processes the event triggered after a line got collected.
     *
     * This event will be called once for each line regardless its line type.
     * Line type being one of the {@see \Siel\Acumulus\Data\LineType} constants.
     */
    public function lineCollectAfter(Line $line, PropertySources $propertySources): void
    {
        /** @var \Siel\Acumulus\Invoice\Source $source */
        $source = $propertySources->get('source');
        /** @var \Siel\Acumulus\Data\Invoice $invoice */
        $invoice = $propertySources->get('invoice');

        // Here you can make changes to the collected line based on your situation,
        // as well as remove added propertySources or prevent adding this line after all.
        $this->getAcumulusContainer()->getLog()->debug(__METHOD__ . '(type=%s, order-id=%d)', $line->getType(), $source->getId());
        $propertySources->remove('myObject');
    }

    /**
     * Processes the event triggered after the raw invoice has been collected, but before
     * it will be completed.
     *
     * Note that the invoice needs yet to be completed:
     * - Fields that depend on configuration are yet to be filled in.
     * - Child lines are still hierarchical.
     * - Foreign amounts are not yet converted to euros.
     * - Amounts may be missing, e.g. only unitpriceinc' and 'vatamount' are filled in,
     *   while unitprice and vatrate are not yet derived.
     */
    public function invoiceCollectAfter(Invoice $invoice, Source $invoiceSource, InvoiceAddResult $localResult): void
    {
        // Here you can make changes to the raw invoice based on your situation.
        $this->getAcumulusContainer()->getLog()->debug(__METHOD__ . '(order_id=%d)', $invoiceSource->getId());
    }

    /**
     * Processes the event triggered after the invoice has been created,
     * that is: collected and completed.
     */
    public function invoiceCreateAfter(Invoice $invoice, Source $invoiceSource, InvoiceAddResult $localResult): void
    {
        // Here you can make changes to the invoice based on your situation.
        $this->getAcumulusContainer()->getLog()->debug(__METHOD__ . '(order_id=%d)', $invoiceSource->getId());
    }

    /**
     * Processes the event triggered before the invoice gets sent.
     *
     * Here you can:
     * - Make final changes to the invoice.
     * - Prevent sending the invoice after all.
     * - Inject custom behaviour (without changing the invoice or localResult) just before
     *   sending.
     */
    public function invoiceSendBefore(Invoice $invoice, InvoiceAddResult $localResult): void
    {
        // Here you can make changes to the completed invoice based on your situation,
        $this->getAcumulusContainer()->getLog()->debug(__METHOD__ . '(order_id=%d)', $invoice->metadataGet(Meta::SourceId));
    }

    /**
     * Processes the event triggered after an invoice has been sent to Acumulus.
     *
     *  Here you can:
     *  - React to the result of sending the invoice, e.g:
     *    - The token gives you access to the Acumulus invoice or packing slip PDF.
     *    - The entry-id can be used to correlate a payment to the invoice.
     *  - Inject custom behaviour (without changing the invoice or localResult) based on
     *    the result of the sending.
     */
    public function invoiceSendAfter(Invoice $invoice, Source $invoiceSource, InvoiceAddResult $result): void
    {
        // Here you can react to the result of sending the invoice to Acumulus
        $logMessage = sprintf(__METHOD__ . '(order_id=%d): ', $invoiceSource->getId());
        if ($result->getMessages(Severity::Exception)) {
            $this->getAcumulusContainer()->getLog()->debug($logMessage . $result->getMessages(Severity::Exception)[0]->getText());
        } elseif ($result->hasError()) {
            // Invoice was sent to Acumulus but not created due to (an) error(s)
            // in the invoice.
            $this->getAcumulusContainer()->getLog()->debug($logMessage . $result->getMessages(Severity::Error)[0]->getText());
        } else {
            // Sent successfully, invoice has been created in Acumulus:
            if ($result->getMessages(Severity::Warning)) {
                // With warnings.
                $this->getAcumulusContainer()->getLog()->debug($logMessage . $result->getMessages(Severity::Warning)[0]->getText());
            } else {
                // Without warnings.
                $this->getAcumulusContainer()->getLog()->debug($logMessage . 'success');
            }

            // Check if an entry id was created.
            $acumulusInvoice = $result->getMainApiResponse();
            if (!empty($acumulusInvoice['entryid'])) {
                // The invoice has been added.
                $this->getAcumulusContainer()->getLog()->debug($logMessage . 'invoice added');
                $invoiceNumber = $acumulusInvoice['invoicenumber'];
                $entryId = $acumulusInvoice['entryid'];
                $token = $acumulusInvoice['token'];
            } elseif (!empty($acumulusInvoice['conceptid'])) {
                // The invoice has been added as a concept.
                $this->getAcumulusContainer()->getLog()->debug($logMessage . 'concept added');
                $conceptId = $acumulusInvoice['conceptid'];
            } else {
                // The invoice was sent in test modus: no invoice nor a concept has been added.
                $this->getAcumulusContainer()->getLog()->debug($logMessage . 'sent in test modus');
            }
        }
    }
}

// Entry point for WP: create and bootstrap our module.
AcumulusCustomiseInvoice::create()->bootstrap();
